//	Maze4DRenderer.swift
//
//	© 2025 by Jeff Weeks
//	See TermsOfUse.txt

import Metal


let tubeRadius = 0.046875			//	or 0.12 to render the app icon
let nodeRadius = tubeRadius
let sliderRadius = 2.5 * tubeRadius	//	or (1.5 * tubeRadius) to render the app icon

//	Goal window width as fraction of total goal width
let goalWindowFraction = 0.75		//	dimensionless

//	The nodeAndTubeRefinementLevel determines the quality of the tubes and their endcaps,
//	while the sliderRefinementLevel determines the quality of the slider ball.
//	The maze nodes look nice and smooth at nodeAndTubeRefinementLevel = 3,
//	but the slider looks better at sliderRefinementLevel = 4.
//
//	In practical terms, a node with nodeAndTubeRefinementLevel = L
//	intersects the coordinate planes in regular 2^(2+L)-gons,
//	which match up exactly to a tube which is a 2^(2+L)-gonal prism.
//
let nodeAndTubeRefinementLevel = 3	//	or higher for high-resolution screenshots
let sliderRefinementLevel = 4		//	or higher for high-resolution screenshots

//	Using 16-bit integers to index mesh vertices works great in 4D Maze,
//	and makes good use of the GPU's power-efficient 16-bit arithmetic.
//	32-bit indices would be needed only for meshes with 2¹⁶ = 65536 or more vertices.
typealias Maze4DMeshIndexType = UInt16	//	UInt16 or UInt32

//	The maze itself gets drawn into a cube with corners at (±1.0, ±1.0, ±1.0),
//	which is then enclosed in a box with corners at (±boxSize, ±boxSize, ±boxSize).
//	In its unrotated position, the box's near face lies flush with the plane
//	of the display, and exactly fills the display's width (in portrait orientation)
//	or height (in landscape orientation).
let boxSize = 1.5

//	The perspectiveFactor gives the ratio
//
//		   distance from observer to center of screen
//		------------------------------------------------
//		distance from edge of screen to center of screen
//
//	It's needed to draw the scene and also to interpret touches.
//
let perspectiveFactor = 3.0

//	Hues run along a scale
//
//		 0	= red
//		1/6	= yellow
//		1/3	= green
//		1/2	= cyan
//		2/3	= blue
//		5/6 = magenta
//		 1	= red
//
//	hueKata and hueAna specify the hues for the most kataward and anaward points.
let hueKata = 0.00	//	red
let hueAna  = 0.75	//	violet


class Maze4DRenderer: GeometryGamesRenderer<Maze4DModel> {

	//	Before calling the superclass's (GeometryGamesRenderer's) init()
	//	to set up itsDevice, itsColorPixelFormat, itsCommandQueue, etc.,
	//	and indeed even before calling any of our own methods to set up
	//	the pipeline states, the mesh set, etc., we must ensure that
	//	all our instances variables have values.  For that reason,
	//	we declare them to be optionals, which are automatically initialized
	//	to nil.  On the other hand, unwrapping such optionals quickly
	//	becomes tedious, and makes the code harder to read (especially
	//	in more complex apps).  So let's define them as
	//	implicitly unwrapped optionals.  I'm slightly uncomfortable
	//	using implicitly unwrapped optionals -- I hadn't intended
	//	to use them at all, for run-time safety -- but in this case
	//	all we need to do to avoid problems is make sure that
	//	our own init() function provides non-nil values for these
	//	instances variables before returning.

	var itsRGBPipelineState:  MTLRenderPipelineState!
	var itsHuesPipelineState: MTLRenderPipelineState!
	var itsDepthStencilState: MTLDepthStencilState!

	var itsNodeMesh:   GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>!
	var itsTubeMesh:   GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>!
	var itsSliderMesh: GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>!
	var itsGoalMesh:   GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>!
	var itsBoxMesh:    GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>?
			//	itsBoxMesh is non-nil only when the user has selected a DifficultyLevel.
		
	//	We'll replace itsNodeInstances, itsMonochromeTubeInstances
	//	and itsRainbowTubeInstances when and only when
	//	the user requests a new maze or changes the shear.
	//	When the app first launches, all three of these variables are nil.
	var itsNodeInstances: MTLBuffer?			//	buffer with array of Maze4DInstanceDataRGB
	var itsMonochromeTubeInstances: MTLBuffer?	//	buffer with array of Maze4DInstanceDataRGB
	var itsRainbowTubeInstances: MTLBuffer?		//	buffer with array of Maze4DInstanceDataHues

	//	Unlike 4D Draw, which uses separate Draw4DVertexRGB and Draw4DVertexHSV
	//	structures, 4D Maze uses a single all-purpose Maze4DVertex structure,
	//	so it can generate a single "tube" buffer, which may be used with
	//	itsRGBPipelineState for monochrome edges or with itsHuesPipelineState
	//	for rainbow edges.  itsRGBPipelineState calls Maze4DVertexFunctionRGB,
	//	which ultimately ignores the wgt value.
	//
	struct Maze4DVertex {
		var pos: SIMD3<Float32>	//	position (x,y,z)
		var nor: SIMD3<Float16>	//	unit-length normal vector (nx, ny, nz)
		var wgt: Float16		//	weight ∈ {0.0, 1.0} used to select between the two tube endpoint hues
	}


// MARK: -
// MARK: Initialization

	init?() {
	
		//	Specify clear-color values in linear extended-range sRGB.
		let theClearColor = MTLClearColor(red: 0.0, green: 0.0, blue: 0.0, alpha: 1.0)

		super.init(
			wantsMultisampling: true,
			wantsDepthBuffer: true,
			clearColor: theClearColor)

		if !(
			setUpPipelineStates()
		 && setUpDepthStencilState()
		 && setUpMeshes()
		) {
			assertionFailure("Unable to set up Metal resources")
			return nil
		}
	}
	
	func setUpPipelineStates() -> Bool {
	
		guard let theGPUFunctionLibrary = itsDevice.makeDefaultLibrary() else {
			assertionFailure("Failed to create default GPU function library")
			return false
		}

		guard let theGPUVertexFunctionRGB = theGPUFunctionLibrary.makeFunction(
				name: "Maze4DVertexFunctionRGB") else {
			assertionFailure("Failed to load Maze4DVertexFunctionRGB")
			return false
		}

		guard let theGPUFragmentFunctionRGB = theGPUFunctionLibrary.makeFunction(
				name: "Maze4DFragmentFunctionRGB") else {
			assertionFailure("Failed to load Maze4DFragmentFunctionRGB")
			return false
		}

		guard let theGPUVertexFunctionHues = theGPUFunctionLibrary.makeFunction(
				name: "Maze4DVertexFunctionHues") else {
			assertionFailure("Failed to load Maze4DVertexFunctionHues")
			return false
		}

		guard let theGPUFragmentFunctionHues = theGPUFunctionLibrary.makeFunction(
				name: "Maze4DFragmentFunctionHues") else {
			assertionFailure("Failed to load Maze4DFragmentFunctionHues")
			return false
		}


		//	Create a MTLVertexDescriptor describing the vertex attributes
		//	that the GPU sends along to each vertex.  Each vertex attribute
		//	may be a 1-, 2-, 3- or 4-component vector.
		
		let theVertexDescriptor = MTLVertexDescriptor()
		
			//	Say where to find each attribute.

		guard let thePosOffset = MemoryLayout.offset(of: \Maze4DVertex.pos) else {
			assertionFailure("Failed to find offset of Maze4DVertex.pos")
			return false
		}
		theVertexDescriptor.attributes[VertexAttributePosition].format = MTLVertexFormat.float3
		theVertexDescriptor.attributes[VertexAttributePosition].bufferIndex = BufferIndexVertexAttributes
		theVertexDescriptor.attributes[VertexAttributePosition].offset = thePosOffset

		guard let theNorOffset = MemoryLayout.offset(of: \Maze4DVertex.nor) else {
			assertionFailure("Failed to find offset of Maze4DVertex.nor")
			return false
		}
		theVertexDescriptor.attributes[VertexAttributeNormal].format = MTLVertexFormat.half3
		theVertexDescriptor.attributes[VertexAttributeNormal].bufferIndex = BufferIndexVertexAttributes
		theVertexDescriptor.attributes[VertexAttributeNormal].offset = theNorOffset

		guard let theWgtOffset = MemoryLayout.offset(of: \Maze4DVertex.wgt) else {
			assertionFailure("Failed to find offset of Maze4DVertex.wgt")
			return false
		}
		theVertexDescriptor.attributes[VertexAttributeWeight].format = MTLVertexFormat.half
		theVertexDescriptor.attributes[VertexAttributeWeight].bufferIndex = BufferIndexVertexAttributes
		theVertexDescriptor.attributes[VertexAttributeWeight].offset = theWgtOffset

			//	Say how to step through each buffer.

		theVertexDescriptor.layouts[BufferIndexVertexAttributes].stepFunction = MTLVertexStepFunction.perVertex
		theVertexDescriptor.layouts[BufferIndexVertexAttributes].stride = MemoryLayout<Maze4DVertex>.stride
		
		//	Create the MTLRenderPipelineDescriptors.
		//
		//		Note:  I could probably reuse the same MTLRenderPipelineDescriptor,
		//		changing only the GPU functions, to create both MTLRenderPipelineStates,
		//		but the documentation doesn't guarantee that this is allowed,
		//		so let's play it safe and create separate MTLRenderPipelineDescriptors.
		//
		
		//	RGB render pipeline
		
		let theRGBPipelineDescriptor = MTLRenderPipelineDescriptor()
		theRGBPipelineDescriptor.label = "4D Maze RGB render pipeline"
		theRGBPipelineDescriptor.rasterSampleCount = itsSampleCount
		theRGBPipelineDescriptor.vertexFunction = theGPUVertexFunctionRGB
		theRGBPipelineDescriptor.fragmentFunction = theGPUFragmentFunctionRGB
		theRGBPipelineDescriptor.vertexDescriptor = theVertexDescriptor
		theRGBPipelineDescriptor.colorAttachments[0].pixelFormat = itsColorPixelFormat
		theRGBPipelineDescriptor.depthAttachmentPixelFormat = itsDepthPixelFormat
		theRGBPipelineDescriptor.stencilAttachmentPixelFormat = MTLPixelFormat.invalid

		do {
			itsRGBPipelineState = try itsDevice.makeRenderPipelineState(descriptor: theRGBPipelineDescriptor)
		} catch {
			assertionFailure("Couldn't create itsRGBPipelineState: \(error)")
			return false
		}
		
		//	Hues render pipeline
		
		let theHuesPipelineDescriptor = MTLRenderPipelineDescriptor()
		theHuesPipelineDescriptor.label = "4D Maze hues render pipeline"
		theHuesPipelineDescriptor.rasterSampleCount = itsSampleCount
		theHuesPipelineDescriptor.vertexFunction = theGPUVertexFunctionHues
		theHuesPipelineDescriptor.fragmentFunction = theGPUFragmentFunctionHues
		theHuesPipelineDescriptor.vertexDescriptor = theVertexDescriptor
		theHuesPipelineDescriptor.colorAttachments[0].pixelFormat = itsColorPixelFormat
		theHuesPipelineDescriptor.depthAttachmentPixelFormat = itsDepthPixelFormat
		theHuesPipelineDescriptor.stencilAttachmentPixelFormat = MTLPixelFormat.invalid

		do {
			itsHuesPipelineState = try itsDevice.makeRenderPipelineState(descriptor: theHuesPipelineDescriptor)
		} catch {
			assertionFailure("Couldn't create itsHuesPipelineState: \(error)")
			return false
		}

		return true
	}
	
	func setUpDepthStencilState() -> Bool {

		//	init() has already checked that itsDepthPixelFormat != MTLPixelFormat.invalid
		
		let theDepthStencilDescriptor = MTLDepthStencilDescriptor()
		theDepthStencilDescriptor.depthCompareFunction = MTLCompareFunction.less
		theDepthStencilDescriptor.isDepthWriteEnabled = true

		itsDepthStencilState = itsDevice.makeDepthStencilState(
									descriptor: theDepthStencilDescriptor)
		if itsDepthStencilState == nil {
			assertionFailure("Couldn't create itsDepthStencilState")
			return false
		}

		return true
	}

	func setUpMeshes() -> Bool {

		itsNodeMesh = makeBallMesh(
						radius: nodeRadius,
						refinementLevel: nodeAndTubeRefinementLevel)

		itsTubeMesh = makeTubeMesh(
						radius: tubeRadius,
						refinementLevel: nodeAndTubeRefinementLevel)

		itsSliderMesh = makeBallMesh(
						radius: sliderRadius,
						refinementLevel: sliderRefinementLevel)

		itsGoalMesh = makeGoalMesh(
						sliderRadius: sliderRadius)

		//	Leave itsBoxMesh = nil until the user chooses a difficulty level.
		
		return true
	}

	func makeBallMesh(
		radius: Double,
		refinementLevel: Int
	) -> GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType> {

		//	Create a unit sphere.
		let theUnitSphereMesh: GeometryGamesMesh<GeometryGamesPlainVertex, Maze4DMeshIndexType>
			= unitSphereMesh(
				baseShape: .octahedron,
				refinementLevel: refinementLevel)

		let theBallVertices = theUnitSphereMesh.itsVertices.map() { v in

			//	Convert from GeometryGamesPlainVertex to Maze4DVertex
			//	while also rescaling the ball.
			
			Maze4DVertex(
				pos: Float32(radius) * v.pos,
				nor: v.nor,
				wgt: 0.0)	//	balls don't use wgt
		}

		let theBallIndices = theUnitSphereMesh.itsIndices	//	same facet indices

		let theBallMesh = GeometryGamesMesh(
							vertices: theBallVertices,
							indices: theBallIndices)

		let theBallMeshBuffers = theBallMesh.makeBuffers(device: itsDevice)

		return theBallMeshBuffers
	}

	func makeTubeMesh(
		radius: Double,
		refinementLevel: Int
	) -> GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>
		//	Returns a cylinder with of radius tubeRadius
		//	and ends at z = 0.0 and z = 1.0
	{

		//	Create a unit cylinder.
		let theUnitCylinderMesh: GeometryGamesMesh<GeometryGamesPlainVertex, Maze4DMeshIndexType>
			= unitCylinderMesh(
				style: .prism,
				refinementLevel: refinementLevel)

		let theTubeVertices = theUnitCylinderMesh.itsVertices.map() { v in
		
			//	Convert from GeometryGamesPlainVertex to Maze4DVertex
			//	while also rescaling the cylinder.
			
			Maze4DVertex(

				pos: SIMD3<Float32>(
						Float32(tubeRadius) * v.pos[0],
						Float32(tubeRadius) * v.pos[1],
						0.5 * (v.pos[2] + 1.0)	//	∈ {0.0, 1.0}
					 ),

				nor: v.nor,

				//	The sign of the z component of theCylinderVertices
				//	determines the vertex's "weight" ∈ {0.0, 1.0}, which
				//	gets used to select between the two tube endpoint colors.
				wgt: Float16(v.pos[2] > 0.0 ? 0.0 : 1.0)
			)
		}

		let theTubeIndices = theUnitCylinderMesh.itsIndices	//	same facet indices

		let theTubeMesh = GeometryGamesMesh(
							vertices: theTubeVertices,
							indices: theTubeIndices)

		let theTubeMeshBuffers = theTubeMesh.makeBuffers(device: itsDevice)

		return theTubeMeshBuffers
	}
	
	func makeGoalMesh(
		sliderRadius: Double
	) -> GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType> {

		//	Let the size of the goal be such that
		//	the slider fits snugly within the goal's window.
		//
		let theGoalHalfWidth
			= sliderRadius / sqrt(1.0 * 1.0  +  goalWindowFraction * goalWindowFraction)
		let theWindowHalfWidth = goalWindowFraction * theGoalHalfWidth

		let g = Float32(theGoalHalfWidth)
		let w = Float32(theWindowHalfWidth)
		let theVertices = [

			Maze4DVertex(pos: SIMD3<Float32>(-g, -g, -g), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, -w, -w), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +g, -g), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +w, -w), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +g, +g), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +w, +w), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, -g, +g), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, -w, +w), nor: SIMD3<Float16>(-1.0,  0.0,  0.0), wgt: 0.0),

			Maze4DVertex(pos: SIMD3<Float32>(+g, -w, -w), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -g, -g), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +w, -w), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +g, -g), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +w, +w), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +g, +g), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -w, +w), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -g, +g), nor: SIMD3<Float16>(+1.0,  0.0,  0.0), wgt: 0.0),

			Maze4DVertex(pos: SIMD3<Float32>(-g, -g, -g), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-w, -g, -w), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, -g, +g), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-w, -g, +w), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -g, +g), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, -g, +w), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -g, -g), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, -g, -w), nor: SIMD3<Float16>( 0.0, -1.0,  0.0), wgt: 0.0),

			Maze4DVertex(pos: SIMD3<Float32>(-w, +g, -w), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +g, -g), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-w, +g, +w), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +g, +g), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, +g, +w), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +g, +g), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, +g, -w), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +g, -g), nor: SIMD3<Float16>( 0.0, +1.0,  0.0), wgt: 0.0),

			Maze4DVertex(pos: SIMD3<Float32>(-g, -g, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-w, -w, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -g, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, -w, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +g, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, +w, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +g, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-w, +w, -g), nor: SIMD3<Float16>( 0.0,  0.0, -1.0), wgt: 0.0),

			Maze4DVertex(pos: SIMD3<Float32>(-w, -w, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, -g, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, -w, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, -g, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+w, +w, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(+g, +g, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-w, +w, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0),
			Maze4DVertex(pos: SIMD3<Float32>(-g, +g, +g), nor: SIMD3<Float16>( 0.0,  0.0, +1.0), wgt: 0.0)
		]

		let theOuterFacets: [Maze4DMeshIndexType] = [

			//	Winding order is clockwise when viewed
			//	from outside the goal in a left-handed coordinate system.

			 0,  1,  2,
			 1,  3,  2,
			 2,  3,  4,
			 3,  5,  4,
			 4,  5,  6,
			 5,  7,  6,
			 6,  7,  0,
			 7,  1,  0,

			 8,  9, 10,
			 9, 11, 10,
			10, 11, 12,
			11, 13, 12,
			12, 13, 14,
			13, 15, 14,
			14, 15,  8,
			15,  9,  8,

			16, 17, 18,
			17, 19, 18,
			18, 19, 20,
			19, 21, 20,
			20, 21, 22,
			21, 23, 22,
			22, 23, 16,
			23, 17, 16,

			24, 25, 26,
			25, 27, 26,
			26, 27, 28,
			27, 29, 28,
			28, 29, 30,
			29, 31, 30,
			30, 31, 24,
			31, 25, 24,

			32, 33, 34,
			33, 35, 34,
			34, 35, 36,
			35, 37, 36,
			36, 37, 38,
			37, 39, 38,
			38, 39, 32,
			39, 33, 32,

			40, 41, 42,
			41, 43, 42,
			42, 43, 44,
			43, 45, 44,
			44, 45, 46,
			45, 47, 46,
			46, 47, 40,
			47, 41, 40
		]
		let theInnerFacets = (0 ..< 144).map() { i in
		
			//	Reverse the winding order of each facet's vertices.
			//
			//	Inner facets will be dark, because the vertices' normal vectors
			//	"point the wrong way".  This is the desired result,
			//	to provide good contrast between inner and outer facets.
			//
			switch i%3 {
				case 0: return theOuterFacets[ i ]
				case 1: return theOuterFacets[i+1]
				case 2: return theOuterFacets[i-1]
				default: preconditionFailure("Impossible error in makeGoalMesh()")
			}
		}
		let theFacets = theOuterFacets + theInnerFacets

		let theGoalMesh = GeometryGamesMesh<Maze4DVertex, Maze4DMeshIndexType> (
							vertices: theVertices,
							indices: theFacets)

		let theGoalMeshBuffers = theGoalMesh.makeBuffers(device: itsDevice)
		
		return theGoalMeshBuffers
	}


// MARK: -
// MARK: Refresh

	func refreshForNewMaze(
		modelData: Maze4DModel
	) {

		itsNodeInstances = makeNodeInstances(modelData: modelData)
		(itsMonochromeTubeInstances, itsRainbowTubeInstances)
			= makeTubeInstances(modelData: modelData)
		
		itsBoxMesh = makeBoxMesh(difficultyLevel: modelData.itsMaze?.difficultyLevel)
			
		modelData.changeCount += 1
	}

	func refreshForNewShear(
		modelData: Maze4DModel
	) {
	
		itsNodeInstances = makeNodeInstances(modelData: modelData)
		(itsMonochromeTubeInstances, itsRainbowTubeInstances)
			= makeTubeInstances(modelData: modelData)
			
		modelData.changeCount += 1
	}
	
	func makeNodeInstances(
		modelData: Maze4DModel
	) -> MTLBuffer? {	//	buffer with array of Maze4DInstanceDataRGB
	
		guard let theMaze = modelData.itsMaze else {
			return nil
		}
		
		let n = theMaze.difficultyLevel.n
		let theNodeCount = n * n * n * n

		guard let theNodeInstanceBuffer = itsDevice.makeBuffer(
				length: theNodeCount * MemoryLayout<Maze4DInstanceDataRGB>.stride,
				options: .storageModeShared) else {
			return nil
		}
		let theNodeInstances = theNodeInstanceBuffer.contents()
			.bindMemory(to: Maze4DInstanceDataRGB.self, capacity: theNodeCount)
		
		var ι = 0	//	index of the next entry in theNodeInstances
		for l in 0 ..< n {
		
			//	Note the sheared position and color of the node at (0,0,0,l).  //	'l' = "ell", not "one"
			let (the000lPos, _, theRGBColor) = shearedPosition(
												position4D: SIMD4<Double>(0.0, 0.0, 0.0, Double(l)),
												difficulty: theMaze.difficultyLevel,
												shearFactor: modelData.its4DShearFactor)

			for i in 0 ..< n {
				for j in 0 ..< n {
					for k in 0 ..< n {
					
						theNodeInstances[ι].itsModelMatrix = matrix_identity_float4x4
						theNodeInstances[ι].itsModelMatrix[3][0] = Float(the000lPos[0]) + Float(i)
						theNodeInstances[ι].itsModelMatrix[3][1] = Float(the000lPos[1]) + Float(j)
						theNodeInstances[ι].itsModelMatrix[3][2] = Float(the000lPos[2]) + Float(k)
						
						theNodeInstances[ι].itsRGB = theRGBColor
						
						ι += 1
					}
				}
			}
		}
		precondition(ι == theNodeCount, "Found wrong number of nodes")
		
		return theNodeInstanceBuffer
	}
	
	func makeTubeInstances(
		modelData: Maze4DModel
	) -> (MTLBuffer?, MTLBuffer?)
		//	returns optional MTLBuffer containing
		//		an array of Maze4DInstanceDataRGB  for monochrome tubes
		//		an array of Maze4DInstanceDataHues for rainbow tubes
	{
		guard let theMaze = modelData.itsMaze else {
			return (nil, nil)
		}
		
		let n = theMaze.difficultyLevel.n
		let theTubeCount =  n * n * n * n  -  1
		
		//	Count the number of monochrome tubes and rainbow tubes.
		var theMonochromeTubeCount = 0
		var theRainbowTubeCount = 0
		for i in 0 ..< n {
			for j in 0 ..< n {
				for k in 0 ..< n {
					for l in 0 ..< n {
					
						for d in 0 ... 2 {	//	exclude d = 3
							if theMaze.edges[i][j][k][l].outbound[d] {
								theMonochromeTubeCount += 1
							}
						}
						
						if theMaze.edges[i][j][k][l].outbound[3] {
							theRainbowTubeCount += 1
						}
					}
				}
			}
		}
		precondition(
			theMonochromeTubeCount + theRainbowTubeCount == theTubeCount,
			"Internal error:  wrong total number of tubes")
		
		guard
		
			let theMonochromeInstanceBuffer = itsDevice.makeBuffer(
				length: theMonochromeTubeCount * MemoryLayout<Maze4DInstanceDataRGB>.stride,
				options: .storageModeShared),
				
			let theRainbowInstanceBuffer = itsDevice.makeBuffer(
				length: theRainbowTubeCount * MemoryLayout<Maze4DInstanceDataHues>.stride,
				options: .storageModeShared)
		else {
			return (nil, nil)
		}
		
		let theMonochromeInstances = theMonochromeInstanceBuffer.contents()
			.bindMemory(
				to: Maze4DInstanceDataRGB.self,
				capacity: theMonochromeTubeCount)
		
		let theRainbowInstances = theRainbowInstanceBuffer.contents()
			.bindMemory(
				to: Maze4DInstanceDataHues.self,
				capacity: theRainbowTubeCount)

		let rt2 = Float(sqrt( 1.0 / 2.0 ))
		let rt3 = Float(sqrt( 1.0 / 3.0 ))
		let rt6 = Float(sqrt( 1.0 / 6.0 ))
		let r23 = Float(sqrt( 2.0 / 3.0 ))

		let theRotations: [simd_float4x4] = [
		
			//	rotate z-axis to x-axis
			simd_float4x4(
				SIMD4<Float>( 0.0,  0.0, -1.0,  0.0),
				SIMD4<Float>( 0.0,  1.0,  0.0,  0.0),
				SIMD4<Float>( 1.0,  0.0,  0.0,  0.0),
				SIMD4<Float>( 0.0,  0.0,  0.0,  1.0)),
		
			//	rotate z-axis to y-axis
			simd_float4x4(
				SIMD4<Float>( 1.0,  0.0,  0.0,  0.0),
				SIMD4<Float>( 0.0,  0.0, -1.0,  0.0),
				SIMD4<Float>( 0.0,  1.0,  0.0,  0.0),
				SIMD4<Float>( 0.0,  0.0,  0.0,  1.0)),
		
			//	keep z-axis fixed (identity matrix)
			simd_float4x4(
				SIMD4<Float>( 1.0,  0.0,  0.0,  0.0),
				SIMD4<Float>( 0.0,  1.0,  0.0,  0.0),
				SIMD4<Float>( 0.0,  0.0,  1.0,  0.0),
				SIMD4<Float>( 0.0,  0.0,  0.0,  1.0)),
		
			//	rotate z-axis to the diagonal (√⅓, √⅓, √⅓)
			simd_float4x4(
				SIMD4<Float>(-rt2,  rt2,  0.0,  0.0),
				SIMD4<Float>(-rt6, -rt6,  r23,  0.0),
				SIMD4<Float>( rt3,  rt3,  rt3,  0.0),
				SIMD4<Float>( 0.0,  0.0,  0.0,  1.0))
		]

		let theDiagonalCompression = Float( sqrt(3.0) * modelData.its4DShearFactor * tubeRadius )

		var m = 0	//	index of the next entry in theMonochromeInstances
		var r = 0	//	index of the next entry in theRainbowInstances
		for l in 0 ..< n {

			//	Note the sheared positions and colors
			//	of the nodes at (0, 0, 0, l) and (0, 0, 0 , l+1).
			
			let (theShearedPosition0, theHue0, theRGBColor0)
				= shearedPosition(
					position4D: SIMD4<Double>(0.0, 0.0, 0.0, Double(l)),
					difficulty: theMaze.difficultyLevel,
					shearFactor: modelData.its4DShearFactor)

			let (_, theHue1, _)
				= shearedPosition(
					position4D: SIMD4<Double>(0.0, 0.0, 0.0, Double(l+1)),
					difficulty: theMaze.difficultyLevel,
					shearFactor: modelData.its4DShearFactor)

			for i in 0 ..< n {
				for j in 0 ..< n {
					for k in 0 ..< n {

						//	monochrome tubes
						for d in 0 ... 2 {	//	exclude d = 3
							if theMaze.edges[i][j][k][l].outbound[d] {
							
								var theTubePlacement = theRotations[d]

								theTubePlacement[3][0] = Float(theShearedPosition0[0]) + Float(i)
								theTubePlacement[3][1] = Float(theShearedPosition0[1]) + Float(j)
								theTubePlacement[3][2] = Float(theShearedPosition0[2]) + Float(k)
								
								theMonochromeInstances[m].itsModelMatrix = theTubePlacement
								theMonochromeInstances[m].itsRGB = theRGBColor0
								
								m += 1
							}
						}
						
						//	rainbow tubes
						if theMaze.edges[i][j][k][l].outbound[3] {

							var theTubePlacement = theRotations[3]

							//	Monochrome tubes already have the correct length
							//	(namely length one) but each rainbow tube needs
							//	to get compressed in the z-direction.
							theTubePlacement[2][0] *= theDiagonalCompression
							theTubePlacement[2][1] *= theDiagonalCompression
							theTubePlacement[2][2] *= theDiagonalCompression

							theTubePlacement[3][0] = Float(theShearedPosition0[0]) + Float(i)
							theTubePlacement[3][1] = Float(theShearedPosition0[1]) + Float(j)
							theTubePlacement[3][2] = Float(theShearedPosition0[2]) + Float(k)
							
							theRainbowInstances[r].itsModelMatrix = theTubePlacement
							theRainbowInstances[r].itsHues[0] = Float16(theHue0)
							theRainbowInstances[r].itsHues[1] = Float16(theHue1)
							
							r += 1
						}
					}
				}
			}
		}
		precondition(
			m == theMonochromeTubeCount,
			"Found wrong number of monochrome tubes")
		precondition(
			r == theRainbowTubeCount,
			"Found wrong number of rainbow tubes")
		
		return (theMonochromeInstanceBuffer, theRainbowInstanceBuffer)
	}

	func makeBoxMesh(
		difficultyLevel: DifficultyLevel?
	) -> GeometryGamesMeshBuffers<Maze4DVertex, Maze4DMeshIndexType>? {

		//	Technical note:  If we create the box in maze coordinates,
		//	then on the GPU the Maze4DVertexFunctionRGB() can safely apply
		//	itsModelMatrix (the identity!) to normal vectors
		//	with affecting their length.

		guard let theDifficultyLevel = difficultyLevel else {
			return nil
		}

		let n = theDifficultyLevel.n

		//	Box placement in maze coordinates [0, n - 1]
		let theBoxCenter =              0.5 * Double(n - 1)
		let theBoxHalfWidth = boxSize * 0.5 * Double(n - 1)
	
		//	Lower and upper bounds on box coordinates
		let lower = Float32(theBoxCenter - theBoxHalfWidth)
		let upper = Float32(theBoxCenter + theBoxHalfWidth)

		//	Create a mesh for the whole box, rather than just
		//	a single face which would need to get transformed
		//	by 6 matrices.  The extra memory required
		//	to store a mesh for the whole box is negligible
		//	-- as are any speed considerations at render time --
		//	and this approach keeps the rest of the code simple.
	

		//	Vertices

		let theVertices = [
	
			//	x == -1
			Maze4DVertex(pos: [lower, lower, lower], nor: [+1.0,  0.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, upper, lower], nor: [+1.0,  0.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, lower, upper], nor: [+1.0,  0.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, upper, upper], nor: [+1.0,  0.0,  0.0], wgt: 0.0),

			//	x == +1
			Maze4DVertex(pos: [upper, lower, lower], nor: [-1.0,  0.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, lower, upper], nor: [-1.0,  0.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, upper, lower], nor: [-1.0,  0.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, upper, upper], nor: [-1.0,  0.0,  0.0], wgt: 0.0),

			//	y == -1
			Maze4DVertex(pos: [lower, lower, lower], nor: [ 0.0, +1.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, lower, upper], nor: [ 0.0, +1.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, lower, lower], nor: [ 0.0, +1.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, lower, upper], nor: [ 0.0, +1.0,  0.0], wgt: 0.0),

			//	y == +1
			Maze4DVertex(pos: [lower, upper, lower], nor: [ 0.0, -1.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, upper, lower], nor: [ 0.0, -1.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, upper, upper], nor: [ 0.0, -1.0,  0.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, upper, upper], nor: [ 0.0, -1.0,  0.0], wgt: 0.0),

			//	z == -1
			Maze4DVertex(pos: [lower, lower, lower], nor: [ 0.0,  0.0, +1.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, lower, lower], nor: [ 0.0,  0.0, +1.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, upper, lower], nor: [ 0.0,  0.0, +1.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, upper, lower], nor: [ 0.0,  0.0, +1.0], wgt: 0.0),

			//	z == +1
			Maze4DVertex(pos: [lower, lower, upper], nor: [ 0.0,  0.0, -1.0], wgt: 0.0),
			Maze4DVertex(pos: [lower, upper, upper], nor: [ 0.0,  0.0, -1.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, lower, upper], nor: [ 0.0,  0.0, -1.0], wgt: 0.0),
			Maze4DVertex(pos: [upper, upper, upper], nor: [ 0.0,  0.0, -1.0], wgt: 0.0)
		]


		//	Facets

		let theIndices: [Maze4DMeshIndexType] =
		[
			 0,  1,  2,   2,  1,  3,
			 4,  5,  6,   6,  5,  7,
			 8,  9, 10,  10,  9, 11,
			12, 13, 14,  14, 13, 15,
			16, 17, 18,  18, 17, 19,
			20, 21, 22,  22, 21, 23
		]


		let theBoxMesh = GeometryGamesMesh<Maze4DVertex, Maze4DMeshIndexType> (
							vertices: theVertices,
							indices: theIndices)

		let theBoxMeshBuffers = theBoxMesh.makeBuffers(device: itsDevice)
		
		return theBoxMeshBuffers
	}


// MARK: -
// MARK: Rendering

	//	GeometryGamesRenderer (our superclass) provides the functions
	//
	//		render()
	//			for standard onscreen rendering
	//
	//		createOffscreenImage()
	//			for copyImage and saveImage
	//
	//	Those two functions handle the app-independent parts
	//	of rendering an image, but for the app-dependent parts
	//	they call encodeCommands(), which we override here
	//	to provide the app-specific content.

	override func encodeCommands(
		modelData: Maze4DModel,
		commandBuffer: MTLCommandBuffer,
		renderPassDescriptor: MTLRenderPassDescriptor,
		frameWidth: Int,
		frameHeight: Int,
		transparentBackground: Bool,	//	Only KaleidoPaint and KaleidoTile need to know
										//	transparentBackground while encoding commands.
		extraRenderFlag: Bool?,
		quality: GeometryGamesImageQuality
	) {

		//	When the user first launches the app, no maze will be present.
		guard let theMaze = modelData.itsMaze else {

			//  Submit an empty render encoder so that Metal will clear the framebuffer.
			if let theCommandEncoder = commandBuffer.makeRenderCommandEncoder(
					descriptor: renderPassDescriptor) {
	
				theCommandEncoder.endEncoding()
			}

			return
		}

		var theUniformData = prepareUniformData(
								modelData: modelData,
								maze: theMaze,
								frameWidth: frameWidth,
								frameHeight: frameHeight)

		if let theCommandEncoder = commandBuffer.makeRenderCommandEncoder(
			descriptor: renderPassDescriptor) {
					
			theCommandEncoder.label = "4D Maze render encoder"

			theCommandEncoder.setDepthStencilState(itsDepthStencilState)
			theCommandEncoder.setCullMode(MTLCullMode.back)
			theCommandEncoder.setFrontFacing(MTLWinding.clockwise)	//	the default
			theCommandEncoder.setVertexBytes(
								&theUniformData,
								length: MemoryLayout.size(ofValue: theUniformData),
								index: BufferIndexUniforms)


			//	draw monochrome stuff

			theCommandEncoder.setRenderPipelineState(itsRGBPipelineState)

			//	nodes
			if let theNodeInstances = itsNodeInstances {
			
				let theInstanceStride = MemoryLayout<Maze4DInstanceDataRGB>.stride
				precondition(
					theNodeInstances.length % theInstanceStride == 0,
					"theNodeInstances buffer size isn't an integer multiple of theInstanceStride")
				let theNodeInstanceCount = theNodeInstances.length / theInstanceStride
				
				theCommandEncoder.pushDebugGroup("Draw nodes")
				
				theCommandEncoder.setVertexBuffer(theNodeInstances,
									offset: 0,
									index: BufferIndexInstanceData)
				theCommandEncoder.setVertexBuffer(itsNodeMesh.vertexBuffer,
									offset: 0,
									index: BufferIndexVertexAttributes)

				theCommandEncoder.drawIndexedPrimitives(
					type: .triangle,
					indexCount: itsNodeMesh.indexCount,
					indexType: itsNodeMesh.mtlIndexType,
					indexBuffer: itsNodeMesh.indexBuffer,
					indexBufferOffset: 0,
					instanceCount: theNodeInstanceCount)

				theCommandEncoder.popDebugGroup()
			}

			//	monochrome tubes
			if let theMonochromeTubeInstances = itsMonochromeTubeInstances {
			
				let theInstanceStride = MemoryLayout<Maze4DInstanceDataRGB>.stride
				precondition(
					theMonochromeTubeInstances.length % theInstanceStride == 0,
					"theMonochromeTubeInstances buffer size isn't an integer multiple of theInstanceStride")
				let theMonochromeTubeInstanceCount
					= theMonochromeTubeInstances.length / theInstanceStride
			
				theCommandEncoder.pushDebugGroup("Draw monochrome tubes")
				
				theCommandEncoder.setVertexBuffer(theMonochromeTubeInstances,
									offset: 0,
									index: BufferIndexInstanceData)
				theCommandEncoder.setVertexBuffer(itsTubeMesh.vertexBuffer,
									offset: 0,
									index: BufferIndexVertexAttributes)

				theCommandEncoder.drawIndexedPrimitives(
					type: .triangle,
					indexCount: itsTubeMesh.indexCount,
					indexType: itsTubeMesh.mtlIndexType,
					indexBuffer: itsTubeMesh.indexBuffer,
					indexBufferOffset: 0,
					instanceCount: theMonochromeTubeInstanceCount)

				theCommandEncoder.popDebugGroup()
			}

			//	slider
			do {
			
				theCommandEncoder.pushDebugGroup("Draw slider")
				
				theCommandEncoder.setVertexBytes(
					&theUniformData.itsSliderInstanceData,
					length: MemoryLayout.size(ofValue: theUniformData.itsSliderInstanceData),
					index: BufferIndexInstanceData)
				theCommandEncoder.setVertexBuffer(itsSliderMesh.vertexBuffer,
					offset: 0,
					index: BufferIndexVertexAttributes)

				theCommandEncoder.drawIndexedPrimitives(
					type: .triangle,
					indexCount: itsSliderMesh.indexCount,
					indexType: itsSliderMesh.mtlIndexType,
					indexBuffer: itsSliderMesh.indexBuffer,
					indexBufferOffset: 0,
					instanceCount: 1)

				theCommandEncoder.popDebugGroup()
			}

			//	goal
			do {

				theCommandEncoder.pushDebugGroup("Draw goal")
				
				theCommandEncoder.setVertexBytes(
					&theUniformData.itsGoalInstanceData,
					length: MemoryLayout.size(ofValue: theUniformData.itsGoalInstanceData),
					index: BufferIndexInstanceData)
				theCommandEncoder.setVertexBuffer(itsGoalMesh.vertexBuffer,
					offset: 0,
					index: BufferIndexVertexAttributes)

				theCommandEncoder.drawIndexedPrimitives(
					type: .triangle,
					indexCount: itsGoalMesh.indexCount,
					indexType: itsGoalMesh.mtlIndexType,
					indexBuffer: itsGoalMesh.indexBuffer,
					indexBufferOffset: 0,
					instanceCount: 1)

				theCommandEncoder.popDebugGroup()
			}

			//	box
			if let theBoxMesh = itsBoxMesh {
			
				theCommandEncoder.pushDebugGroup("Draw box")

				theCommandEncoder.setVertexBytes(
					&theUniformData.itsBoxInstanceData,
					length: MemoryLayout.size(ofValue: theUniformData.itsBoxInstanceData),
					index: BufferIndexInstanceData)
				theCommandEncoder.setVertexBuffer(theBoxMesh.vertexBuffer,
					offset: 0,
					index: BufferIndexVertexAttributes)

				theCommandEncoder.drawIndexedPrimitives(
					type: .triangle,
					indexCount: theBoxMesh.indexCount,
					indexType: theBoxMesh.mtlIndexType,
					indexBuffer: theBoxMesh.indexBuffer,
					indexBufferOffset: 0,
					instanceCount: 1)

				theCommandEncoder.popDebugGroup()
			}

			//	draw rainbow stuff

			theCommandEncoder.setRenderPipelineState(itsHuesPipelineState)

			//	rainbow tubes
			if let theRainbowTubeInstances = itsRainbowTubeInstances {
			
				let theInstanceStride = MemoryLayout<Maze4DInstanceDataHues>.stride
				precondition(
					theRainbowTubeInstances.length % theInstanceStride == 0,
					"theRainbowTubeInstances buffer size isn't an integer multiple of theInstanceStride")
				let theRainbowTubeInstanceCount
					= theRainbowTubeInstances.length / theInstanceStride
			
				theCommandEncoder.pushDebugGroup("Draw rainbow tubes")
				
				theCommandEncoder.setVertexBuffer(theRainbowTubeInstances,
									offset: 0,
									index: BufferIndexInstanceData)
				theCommandEncoder.setVertexBuffer(itsTubeMesh.vertexBuffer,
									offset: 0,
									index: BufferIndexVertexAttributes)

				theCommandEncoder.drawIndexedPrimitives(
					type: .triangle,
					indexCount: itsTubeMesh.indexCount,
					indexType: itsTubeMesh.mtlIndexType,
					indexBuffer: itsTubeMesh.indexBuffer,
					indexBufferOffset: 0,
					instanceCount: theRainbowTubeInstanceCount)

				theCommandEncoder.popDebugGroup()
			}

			theCommandEncoder.endEncoding()
		}
	}

	func prepareUniformData(
		modelData: Maze4DModel,
		maze: MazeData,	//	non-optional MazeData
		frameWidth: Int,
		frameHeight: Int
	) -> Maze4DUniformData {

		//	view matrix
		//
		//		The "view matrix DT" ("DT" = "dilation + translation"
		//		rescales from maze coordinates in the range [0, n - 1]
		//		to box coordinates in the range [-1.0, +1.0], for example
		//
		//			h -> -1.0 + 2.0 * h / (n - 1.0)
		//
		var theViewMatrixDT = matrix_identity_double4x4
		let n = Double(maze.difficultyLevel.n)
		for i in 0...2 {
			theViewMatrixDT[i][i] = 2.0 / (n - 1.0)
			theViewMatrixDT[3][i] = -1.0
		}
		//		The "view matrix R" ("R" = "rotation")
		let theViewMatrixR3x3 = simd_matrix3x3(modelData.itsOrientation)
		let theViewMatrixR = matrix4x4ExtendingMatrix3x3(theViewMatrixR3x3)
		//		Composed view matrix
		let theViewMatrix
			= theViewMatrixR	//	right-to-left matrix composition
			* theViewMatrixDT

		//	projection matrix
		let theProjectionMatrix = makeProjectionMatrix(
									frameWidth: Double(frameWidth),
									frameHeight: Double(frameHeight))

		//	view-projection matrix
		let theViewProjectionMatrix = convertDouble4x4toFloat4x4(
			theProjectionMatrix	//	right-to-left matrix composition
		  * theViewMatrix)

		//	To compute the fog intensity, we conceptually need to compute
		//
		//		         (        )(0)
		//		(x y z 1)(  view  )(0)
		//		         ( matrix )(1)
		//		         (        )(0)
		//
		//	where the column vector (0 0 1 0) serves to extract
		//	the point's z-coordinate in world space.
		//	But rather than passing the whole view matrix
		//	and asking the vertex shader to multiply by it,
		//	it's more efficient to premultiply the view matrix
		//	times (0 0 1 0) and pass that single vector instead.
		//
		let theFogEvaluator = SIMD4<Float16>(
			Float16(theViewMatrix[0][2]),
			Float16(theViewMatrix[1][2]),
			Float16(theViewMatrix[2][2]),
			Float16(theViewMatrix[3][2]) )
		
		let theLightDirection = (	//	points towards the light source
			gMakeScreenshots ?
				SIMD3<Double>(+0.27, +0.17, -0.92) :
				SIMD3<Double>(-0.27, +0.27, -0.92))
							
		
		let theViewerDirection = SIMD3<Double>(0.00,  0.00, -1.00)
								//	points towards the observer
		
		let theSpecularDirection = normalize(theLightDirection + theViewerDirection)
								//	direction of maximal specular reflection
								
		//	The light direction L is an honest vector,
		//	which we write as a column vector for consistency
		//	with SIMD's right-to-left matrix actions.
		//	By contrast a surface normal is, strictly speaking, a 1-form,
		//	which we write as a row vector for consistency
		//	with the right-to-left matrix actions.
		//	To compute the intensity of a light, we take the matrix product
		//	of the light direction with the surface normal:
		//
		//			          (Lx)
		//			(Nx Ny Nz)(Ly)
		//			          (Lz)
		//
		//	(Note:  If we were instead using left-to-right matrix actions,
		//	we'd take the transpose of everything in sight and
		//	write vectors as rows and 1-forms as columns.)
		//
		//	If we rotate the surface and the light source simultaneously
		//	using a matrix M, the light direction vector transforms
		//	as L → ML while the normal transforms as N → NM⁻¹,
		//	leaving their product
		//
		//			          (     )(     )(Lx)
		//			(Nx Ny Nz)( M⁻¹ )(  M  )(Ly)
		//			          (     )(     )(Lz)
		//
		//	invariant, as it must be.
		//
		//	In the present case, however, we want to leave
		//	the light source fixed while we rotate the polyhedron
		//	and its normals.  The product thus becomes
		//
		//			          (     )(Lx)
		//			(Nx Ny Nz)( M⁻¹ )(Ly)
		//			          (     )(Lz)
		//
		//	Conceptually we group the factors as (NM⁻¹)L,
		//	but to minimize the computational load on the GPU,
		//	it's convenient to regroup the factors as N(M⁻¹L)
		//	and pre-compute the product M⁻¹L, which we call
		//	a "normal evaluator".
		//
		//	In general we'd need to invert the matrix M to compute M⁻¹L.
		//	In O(3), however, the inverse is just the transpose.
		//	So instead of computing M⁻¹L we compute its transpose
		//
		//		transpose(M⁻¹L) = transpose(transpose(M) L)
		//						= transpose(L) M
		//
		let theDiffuseEvaluator = SIMD3<Float16>(theLightDirection * theViewMatrixR3x3)

		//	The same logic applies to compute theSpecularEvaluator.
		let theSpecularEvaluator = SIMD3<Float16>(theSpecularDirection * theViewMatrixR3x3)

		//	The slider instance data changes as the user drags the slider,
		//	so keep it as part of the per-frame uniform data,
		//	rather than as a separate fixed buffer.
		let theSliderInstanceData = makeSliderInstance(
				maze: maze,
				shearFactor: modelData.its4DShearFactor,
				sliderBlinkStartTime: modelData.itsSliderBlinkStartTime)
		
		//	The goal's position changes only when the user resets the maze,
		//	but its color changes when the user solves the maze,
		//	and also when it briefly flashes near the beginning
		//	of the first maze in each session.  So let's keep
		//	the goal instance data as part of the per-frame uniform data too.
		let theGoalInstanceData = makeGoalInstance(
				maze: maze,
				shearFactor: modelData.its4DShearFactor,
				gameIsOver: modelData.itsGameIsOver,
				goalFlashStartTime: modelData.itsGoalFlashStartTime)
		
		//	The box's instance data never changes, but nevertheless this seems like
		//	a natural place to include it, along with the slider and goal instance data.
		let theBoxInstanceData = makeBoxInstance()
		
		let theUniformData = Maze4DUniformData(
			itsViewProjectionMatrix: theViewProjectionMatrix,
			itsFogEvaluator: theFogEvaluator,
			itsDiffuseEvaluator: theDiffuseEvaluator,
			itsSpecularEvaluator: theSpecularEvaluator,
			itsSliderInstanceData: theSliderInstanceData,
			itsGoalInstanceData: theGoalInstanceData,
			itsBoxInstanceData: theBoxInstanceData)

		return theUniformData
	}

	func makeProjectionMatrix(
		frameWidth: Double,	//	typically in pixels or points
		frameHeight: Double	//	typically in pixels or points
	) -> simd_double4x4 {	//	returns the projection matrix
	
		if frameWidth <= 0.0 || frameHeight <= 0.0 {
			assertionFailure("nonpositive-size image received")
			return matrix_identity_double4x4
		}
		
		let theIntrinsicUnitsPerPixelOrPoint = intrinsicUnitsPerPixelOrPoint(
													viewWidth: frameWidth,
													viewHeight: frameHeight)
		
		//	Compute the frame's half-width and half-height in intrinsic units.
		let w = 0.5 * frameWidth  * theIntrinsicUnitsPerPixelOrPoint	//	half width
		let h = 0.5 * frameHeight * theIntrinsicUnitsPerPixelOrPoint	//	half height

		//	Assume the observer's eye is (perspectiveFactor * boxSize)
		//	intrinsic units from the center of the display.
		let d = perspectiveFactor * boxSize

		//	In these coordinates, the scene will be centered at d + boxSize,
		//	so let the clipping range run from d - boxSize to d + 3*boxSize
		//	to ensure that the whole box (and a little more) is visible
		//	no matter how it's rotated.
		let n = d - 1.0 * boxSize
		let f = d + 3.0 * boxSize
		precondition(
			n >= 0.01,
			"Near clip plane is too close.")

		precondition(
			w > 0.0 && h > 0.0 && d > 0.0 && n > 0.0 && f > n,
			"invalid projection parameters")

		//	The eight points
		//
		//			(±n*(w/d), ±n*(h/d), n, 1) and
		//			(±f*(w/d), ±f*(h/d), f, 1)
		//
		//	define the view frustum.  More precisely,
		//	because the GPU works in 4D homogeneous coordinates,
		//	it's really a set of eight rays, from the origin (0,0,0,0)
		//	through each of those eight points, that defines
		//	the view frustum as a "hyper-wedge" in 4D space.
		//
		//	Because the GPU works in homogenous coordinates,
		//	we may multiply each point by any scalar constant we like,
		//	without changing the ray that it represents.
		//	So let divide each of those eight points through
		//	by its own z coordinate, giving
		//
		//			(±w/d, ±h/d, 1, 1/n) and
		//			(±w/d, ±h/d, 1, 1/f)
		//
		//	Geometrically, these points define the intersection
		//	of the 4D wedge with the hyperplane z = 1.
		//	Conveniently enough, this intersection is a rectangular box!
		//
		//	Our goal is to find a 4×4 matrix that takes this rectangular box
		//	to the standard clipping box with corners at
		//
		//			(±1, ±1, 0, 1) and
		//			(±1, ±1, 1, 1)
		//
		//	To find such a matrix, let's "follow our nose"
		//	and construct it as the product of several factors.
		//
		//		Note:  Unlike in older versions of the Geometry Games
		//		source code, matrices now act using
		//		the right-to-left (matrix)(column vector) convention,
		//		not the left-to-right (row vector)(matrix) convention.]
		//
		//	Factor #1
		//
		//		The quarter turn matrix
		//
		//			1  0  0  0
		//			0  1  0  0
		//			0  0  0 -1
		//			0  0  1  0
		//
		//		takes the eight points
		//			(±w/d, ±h/d, 1, 1/n) and
		//			(±w/d, ±h/d, 1, 1/f)
		//		to
		//			(±w/d, ±h/d, -1/n, 1) and
		//			(±w/d, ±h/d, -1/f, 1)
		//
		//	Factor #2
		//
		//		The xy dilation
		//
		//			d/w  0   0   0
		//			 0  d/h  0   0
		//			 0   0   1   0
		//			 0   0   0   1
		//
		//		takes
		//			(±w/d, ±h/d, -1/n, 1) and
		//			(±w/d, ±h/d, -1/f, 1)
		//		to
		//			(±1, ±1, -1/n, 1) and
		//			(±1, ±1, -1/f, 1)
		//
		//	Factor #3
		//
		//		The z dilation
		//
		//			1  0  0  0
		//			0  1  0  0
		//			0  0  a  0
		//			0  0  0  1
		//
		//		where a = n*f/(f - n), stretches or shrinks
		//		the box to have unit length in the z direction,
		//		taking
		//			(±1, ±1, -1/n, 1) and
		//			(±1, ±1, -1/f, 1)
		//		to
		//			(±1, ±1, -f/(f - n), 1) and
		//			(±1, ±1, -n/(f - n), 1)
		//
		//	Factor #4
		//
		//		The shear
		//
		//			1  0  0  0
		//			0  1  0  0
		//			0  0  1  b
		//			0  0  0  1
		//
		//		where b = f/(f - n), translates the hyperplane w = 1
		//		just the right amount to take
		//
		//			(±1, ±1, -f/(f - n), 1) and
		//			(±1, ±1, -n/(f - n), 1)
		//		to
		//			(±1, ±1, 0, 1) and
		//			(±1, ±1, 1, 1)
		//
		//		which are the vertices of the standard clipping box,
		//		which is exactly where we wanted to end up.
		//
		//	The projection matrix is the product (taken right-to-left!)
		//	of those four factors:
		//
		//			( 1  0  0  0 )( 1  0  0  0 )( d/w  0   0   0 )( 1  0  0  0 )
		//			( 0  1  0  0 )( 0  1  0  0 )(  0  d/h  0   0 )( 0  1  0  0 )
		//			( 0  0  1  b )( 0  0  a  0 )(  0   0   1   0 )( 0  0  0 -1 )
		//			( 0  0  0  1 )( 0  0  0  1 )(  0   0   0   1 )( 0  0  1  0 )
		//		=
		//			( d/w  0   0   0 )
		//			(  0  d/h  0   0 )
		//			(  0   0   b  -a )
		//			(  0   0   1   0 )
		//

		let theRawProjectionMatrix = simd_double4x4(
			SIMD4<Double>(d/w, 0.0,     0.0,    0.0),
			SIMD4<Double>(0.0, d/h,     0.0,    0.0),
			SIMD4<Double>(0.0, 0.0,   f/(f-n),  1.0),
			SIMD4<Double>(0.0, 0.0, -n*f/(f-n), 0.0))

		//	Let theTranslation be the first factor in the projection matrix
		//	rather than the last factor in the view matrix.
		//
		//	If we translated by d alone, the bounding cube's center
		//	(which is also the coordinate origin) would sit on the display screen.
		//	If we also add in the boxSize, then the bounding cube's front face
		//	will align with the display screen (when the cube's in canonical position).

		var theTranslation = matrix_identity_double4x4
		theTranslation[3][2] = d + boxSize
		
		let theProjectionMatrix
			= theRawProjectionMatrix	//	right-to-left matrix composition
			* theTranslation

		return theProjectionMatrix
	}
	
	func makeSliderInstance(
		maze: MazeData,
		shearFactor: Double,
		sliderBlinkStartTime: CFAbsoluteTime?
	) -> Maze4DInstanceDataRGB {
	
		let (theOffsetCenter, theNominalColor) = sliderOffsetCenterAndColor(
													sliderPosition: maze.sliderPosition,
													difficulty: maze.difficultyLevel,
													shearFactor: shearFactor)
		
		let theColor: simd_half3
		if let theSliderBlinkStartTime = sliderBlinkStartTime {
		
			let theElapsedTime = CFAbsoluteTimeGetCurrent() - theSliderBlinkStartTime
			let theSliderFlashRate = 32.0	//	radians/sec
			let theFlashFactor = Float16(0.5 * (1.0 + cos(theSliderFlashRate * theElapsedTime)))
			let theWhiteColor = simd_half3(1.0, 1.0, 1.0)
			theColor
				=     theFlashFactor     * theNominalColor
				+ (1.0 - theFlashFactor) *  theWhiteColor
			
		} else {
			theColor = theNominalColor
		}
	
		var theModelMatrix = matrix_identity_float4x4
		theModelMatrix[3][0] = Float(theOffsetCenter[0])
		theModelMatrix[3][1] = Float(theOffsetCenter[1])
		theModelMatrix[3][2] = Float(theOffsetCenter[2])
		
		let theSliderInstance = Maze4DInstanceDataRGB(
								itsModelMatrix: theModelMatrix,
								itsRGB: theColor)
		
		return theSliderInstance
	}
	
	func makeGoalInstance(
		maze: MazeData,
		shearFactor: Double,
		gameIsOver: Bool,
		goalFlashStartTime: CFAbsoluteTime?
	) -> Maze4DInstanceDataRGB {
	
		let (theOffsetCenter, _, theNominalColor)
			= shearedPosition(
				position4D: SIMD4<Double>(maze.goalPosition),
				difficulty: maze.difficultyLevel,
				shearFactor: shearFactor)
	
		var theModelMatrix = matrix_identity_float4x4
		theModelMatrix[3][0] = Float(theOffsetCenter[0])
		theModelMatrix[3][1] = Float(theOffsetCenter[1])
		theModelMatrix[3][2] = Float(theOffsetCenter[2])
		
		let theColor = (
			gameIsOver ?
				simd_half3(1.0, 1.0, 1.0) :
				(
					goalFlashStartTime != nil ?
						simd_half3(1.0, 1.0, 1.0) :
						theNominalColor
				)
		)
		
		let theGoalInstance = Maze4DInstanceDataRGB(
								itsModelMatrix: theModelMatrix,
								itsRGB: theColor)
		
		return theGoalInstance
	}

	func makeBoxInstance() -> Maze4DInstanceDataRGB {
	
		let theBoxInstance = Maze4DInstanceDataRGB(
								itsModelMatrix: matrix_identity_float4x4,
								itsRGB: simd_half3(1.0, 1.0, 1.0) )
		
		return theBoxInstance
	}
}


// MARK: -
// MARK: Characteristic size

//	At render time the characteristic size will be used to deduce
//	the view's width and height in intrinsic units.

func characteristicViewSize(
	width: Double,	//	typically in pixels or points
	height: Double	//	typically in pixels or points
) -> Double {

	//	In 4D Maze, a view's "characteristic size" is min(width, height).
	//	In intrinsic units, the characteristic size will always correspond
	//	to the width of the box, even as the user resizes the view and thus
	//	changes the number of pixels lying within it (theCharacteristicSizePp).
	//
	//	Here we compute that same characteristic size in pixels or points.
	//
	//	This is the *only* place that specifies the dependence
	//	of the characteristic size on the view dimensions.
	//	If you want to change the definition, for example
	//	from min(width,height) to sqrt(width*height),
	//	this is the only place you need to do it.
	
	return min(width, height)
}

//	The characteristic size will always correspond
//	to the number of intrinsic units (IU) given below,
//	even as the user resizes the view and thus changes
//	the number of pixels lying within it.
//
let gCharacteristicSizeIU = 2.0 * boxSize

func intrinsicUnitsPerPixelOrPoint(
	viewWidth: Double,	//	typically in pixels or points
	viewHeight: Double	//	typically in pixels or points
) -> Double {	//	number of intrinsic units per pixel or point

	let theCharacteristicSizePp = characteristicViewSize(
									width: viewWidth,
									height: viewHeight)

	let theIntrinsicUnitsPerPixelOrPoint = gCharacteristicSizeIU / theCharacteristicSizePp
	
	return theIntrinsicUnitsPerPixelOrPoint
}
